/*
   Akkutester
   Version vom 09.05.2025
*/

const int RS_PIN = 8;
const int E_PIN = 9;
const int D4_PIN = 10;
const int D5_PIN = 11;
const int D6_PIN = 12;
const int D7_PIN = 13;

const int PIEZO_PIN = 2;
const int RELAIS_PIN = 3;

const int STATUSLED_GN_PIN = 4;
const int STATUSLED_RT_PIN = 5;

const int AUSWAHL_PIN = 6;
const int START_PIN = 7;

const int UAKKU_PIN = A0;
const int UREF_PIN = A1;
const int POL_PIN = A2;

const long UBatterieVoll = 1550;
const long UBatterieLeer = 1500;

const long UAkkuVoll = 1300;
const long UAkkuLeer = 1200;

const long UDefekt = 500;
const long RiMax = 1000;

const char OhmCode = 1;
const char AkkuCode = 2;

const int LedAus = 0;
const int LedGruen = 1;
const int LedRot = 2;
const int LedGelb = 3;
const int LedGruenBlink = 4;
const int LedRotBlink = 8;

const int MessZeit = 20;
const int AnzeigeZeit = 40;

const int LcdZeilen = 2;
const int LcdSpalten = 16;

const boolean AUS = 0;
const boolean EIN = !AUS;

boolean AltStart, Start;
boolean AltAuswahl, Auswahl;
boolean Pol;
boolean Blink;

long U, UR;
long U0, UL, URef;
long Ri;
long ULeer, UVoll;
int ZeitZaehler;

enum { InitMessung,
       StartenMessung,
       StartenWarten,
       MessenU0,
       MessenWarten,
       MessenUL,
       Auswertung,
       AuswertungEnde,
       AnzeigeVoll,
       AnzeigeLeer,
       AnzeigeOk,
       AnzeigeDefekt,
       AnzeigeWarten } Zustand;

byte OhmZeichen[8] = {
  0b01110,
  0b10001,
  0b10001,
  0b10001,
  0b01010,
  0b01010,
  0b11011,
  0b00000
};

byte AkkuZeichen[8] = {
  0b01110,
  0b11111,
  0b10001,
  0b10001,
  0b10001,
  0b10001,
  0b10001,
  0b11111,
};

#include <LiquidCrystal.h>

LiquidCrystal Lcd(RS_PIN, E_PIN, D4_PIN, D5_PIN, D6_PIN, D7_PIN);

void Messungen() {
  InitLcd();
  Lcd.print(F("   Messungen!   "));
  for (;;) {  // Endlosschleife
    digitalWrite(RELAIS_PIN, digitalRead(AUSWAHL_PIN));
    Lcd.setCursor(3, 0);
    Lcd.print('M');
    delay(1);
  }
}

void InitLcd() {
  Lcd.begin(LcdSpalten, LcdZeilen);
  Lcd.createChar(OhmCode, OhmZeichen);
  Lcd.createChar(AkkuCode, AkkuZeichen);
  Lcd.clear();
}

void SchalteStatusLed(int Status) {
  digitalWrite(STATUSLED_GN_PIN, bitRead(Status, 0) || (bitRead(Status, 2) && Blink));
  digitalWrite(STATUSLED_RT_PIN, bitRead(Status, 1) || (bitRead(Status, 3) && Blink));
}

void SchalteRelais(boolean Relais) {
  digitalWrite(RELAIS_PIN, Relais);
}

void ZeigeU0() {
  Lcd.setCursor(0, 0);
  Lcd.print(F("U0="));
  Lcd.print(0.001 * U0);
  Lcd.print(F("V"));
}

void ZeigeUL() {
  Lcd.setCursor(0, 1);
  Lcd.print(F("UL="));
  Lcd.print(0.001 * UL);
  Lcd.print(F("V"));
}

void ZeigeRi() {
  Lcd.setCursor(9, 0);
  if (Ri < 0) {
    Lcd.print(F("Ri=??? "));
  } else if (Ri < 9950) {
    Lcd.print(F("Ri="));
    Lcd.print(0.001 * Ri, 1);
    Lcd.print(OhmCode);
  } else {
    Lcd.print(F("Ri="));
    Lcd.print(0.001 * Ri, 0);
    Lcd.print(OhmCode);
  }
}

void MacheAkkuSymbol(int Ist, int Min, int Max) {
  Ist = map(Ist, Min, Max + 1, 0, 6);
  Ist = constrain(Ist, 0, 5);
  for (int i = 0; i < 5; i++)
    AkkuZeichen[6 - i] = (i < Ist) ? 0x1F : 0x11;
  Lcd.createChar(2, AkkuZeichen);
  Serial.print(F("Akku="));
  Serial.println(Ist);
}

void Eingabe() {
  AltStart = Start;
  AltAuswahl = Auswahl;
  Start = !digitalRead(START_PIN);
  Auswahl = !digitalRead(AUSWAHL_PIN);
  Pol = digitalRead(POL_PIN);
  U = analogRead(UAKKU_PIN);
  UR = analogRead(UREF_PIN);
}

void Verarbeitung() {
  Blink = !Blink;
  if (Pol) {
    if (AltAuswahl != Auswahl) Zustand = InitMessung;
    switch (Zustand) {
      case InitMessung:
        {
          Zustand = StartenMessung;
          break;
        }
      case StartenMessung:
        {
          Zustand = StartenWarten;
          break;
        }
      case StartenWarten:
        {
          if (!AltStart && Start) Zustand = MessenU0;
          ZeitZaehler++;
          break;
        }
      case MessenU0:
        {
          URef = UR;
          U0 = map(U, 0, URef, 0, 4000);
          Serial.print(F("U0="));
          Serial.print(U0);
          Serial.println(F("mV"));
          ZeitZaehler = 0;
          Zustand = MessenWarten;
          break;
        }
      case MessenWarten:
        {
          ZeitZaehler++;
          if (ZeitZaehler == MessZeit) Zustand = MessenUL;
          break;
        }
      case MessenUL:
        {
          UL = map(U, 0, URef, 0, 4000);
          Serial.print(F("UL="));
          Serial.print(UL);
          Serial.println(F("mV"));
          Zustand = Auswertung;
          break;
        }
      case Auswertung:
        {
          if (Auswahl) {
            ULeer = UAkkuLeer;
            UVoll = UAkkuVoll;
          } else {
            ULeer = UBatterieLeer;
            UVoll = UBatterieVoll;
          }
          MacheAkkuSymbol(UL, ULeer, UVoll);
          Ri = (U0 - UL) * 10000 / UL;
          if (Ri > 999000) Ri = 999000;
          if (U0 == 0 || UL == 0) Ri = -1;
          Zustand = AuswertungEnde;
          break;
        }
      case AuswertungEnde:
        {
          Zustand = AnzeigeDefekt;
          if (Ri >= 0) {
            if (Ri < RiMax) Zustand = AnzeigeOk;
            if (Ri < RiMax && U0 >= UVoll) Zustand = AnzeigeVoll;
            if (U0 >= UDefekt && U0 < ULeer) Zustand = AnzeigeLeer;
          }
          break;
        }
      case AnzeigeVoll:
      case AnzeigeLeer:
      case AnzeigeOk:
      case AnzeigeDefekt:
        {
          ZeitZaehler = 0;
          Zustand = AnzeigeWarten;
          break;
        }
      case AnzeigeWarten:
        {
          if (!AltStart && Start) Zustand = MessenU0;
          ZeitZaehler++;
          if (ZeitZaehler == AnzeigeZeit) Zustand = InitMessung;
          break;
        }
    }
  } else Zustand = InitMessung;
}

void VollMelody() {
  tone(PIEZO_PIN, 440);
  delay(200);
  tone(PIEZO_PIN, 550);
  delay(200);
  tone(PIEZO_PIN, 660);
  delay(200);
  noTone(PIEZO_PIN);
}

void LeerMelody() {
  tone(PIEZO_PIN, 440);
  delay(500);
  noTone(PIEZO_PIN);
}

void OkMelody() {
  tone(PIEZO_PIN, 880);
  delay(200);
  tone(PIEZO_PIN, 1320);
  delay(200);
  noTone(PIEZO_PIN);
}

void DefektMelody() {
  tone(PIEZO_PIN, 440);
  delay(200);
  noTone(PIEZO_PIN);
  delay(200);
  tone(PIEZO_PIN, 440);
  delay(200);
  noTone(PIEZO_PIN);
}

void ZeigePruefung() {
  Lcd.setCursor(0, 0);
  if (Auswahl) Lcd.print(F(" Akku"));
  else Lcd.print(F(" Batt."));
  Lcd.print(F("-Pruefung   "));
}

void Ausgabe() {
  if (Pol) {
    switch (Zustand) {
      case InitMessung:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          break;
        }
      case StartenMessung:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          InitLcd();
          ZeigePruefung();
          Lcd.setCursor(0, 1);
          Lcd.print(F(" bitte starten "));
          break;
        }
      case StartenWarten:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          break;
        }
      case MessenU0:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          InitLcd();
          ZeigePruefung();
          Lcd.setCursor(0, 1);
          Lcd.print(F(" laeuft -------"));
          break;
        }
      case MessenWarten:
        {
          SchalteRelais(EIN);
          SchalteStatusLed(LedAus);
          Lcd.setCursor(8 + map(ZeitZaehler, 0, MessZeit, 0, 7), 1);
          Lcd.print(F("*"));
          break;
        }
      case MessenUL:
        {
          SchalteRelais(EIN);
          SchalteStatusLed(LedAus);
          break;
        }
      case Auswertung:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          break;
        }
      case AuswertungEnde:
        {
          InitLcd();
          SchalteRelais(AUS);
          SchalteStatusLed(LedAus);
          ZeigeU0();
          ZeigeUL();
          ZeigeRi();
          Lcd.setCursor(9, 1);
          Lcd.print(AkkuCode);
          break;
        }
      case AnzeigeVoll:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedGruen);
          Lcd.setCursor(11, 1);
          Lcd.print("Voll");
          VollMelody();
          break;
        }
      case AnzeigeLeer:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedGelb);
          Lcd.setCursor(11, 1);
          Lcd.print("Leer");
          LeerMelody();
          break;
        }
      case AnzeigeOk:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedGruen);
          Lcd.setCursor(11, 1);
          Lcd.print("Ok");
          OkMelody();
          break;
        }
      case AnzeigeDefekt:
        {
          SchalteRelais(AUS);
          SchalteStatusLed(LedRot);
          Lcd.setCursor(11, 1);
          Lcd.print("Def.");
          DefektMelody();
          break;
        }
      case AnzeigeWarten:
        {
          SchalteRelais(AUS);
          break;
        }
    }
  } else {
    if (Blink) {
      Lcd.setCursor(0, 0);
      Lcd.print(F("    Falsche     "));
      Lcd.setCursor(0, 1);
      Lcd.print(F("   Polaritaet   "));
      tone(PIEZO_PIN, 1000, 100);
    } else {
      InitLcd();
    }
    SchalteStatusLed(LedRotBlink);
  }
}

void setup() {
  Serial.begin(9600);
  Serial.println(F("EGS-Abschlusspruefung Fruehjahr 2026"));
  pinMode(PIEZO_PIN, OUTPUT);
  pinMode(RELAIS_PIN, OUTPUT);
  pinMode(STATUSLED_GN_PIN, OUTPUT);
  pinMode(STATUSLED_RT_PIN, OUTPUT);
  pinMode(START_PIN, INPUT);
  pinMode(AUSWAHL_PIN, INPUT);
  pinMode(POL_PIN, INPUT);
  digitalWrite(PIEZO_PIN, LOW);
  digitalWrite(RELAIS_PIN, LOW);
  SchalteStatusLed(LedAus);
  InitLcd();
  if (!digitalRead(START_PIN)) Messungen();
  Zustand = InitMessung;
}

void loop() {
  Eingabe();
  Verarbeitung();
  Ausgabe();
  delay(100);
}
